import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { 
  Users, 
  Handshake, 
  DollarSign, 
  TrendingUp,
  FileInput,
  Target
} from "lucide-react";
import type { Lead, Deal, Contact, PipelineStage, DealWithRelations } from "@shared/schema";

export default function Dashboard() {
  const { data: leads, isLoading: leadsLoading } = useQuery<Lead[]>({
    queryKey: ["/api/leads"],
  });

  const { data: deals, isLoading: dealsLoading } = useQuery<DealWithRelations[]>({
    queryKey: ["/api/deals"],
  });

  const { data: contacts, isLoading: contactsLoading } = useQuery<Contact[]>({
    queryKey: ["/api/contacts"],
  });

  const { data: stages } = useQuery<PipelineStage[]>({
    queryKey: ["/api/pipeline-stages"],
  });

  const totalPipelineValue = deals?.reduce((sum, deal) => sum + parseFloat(deal.value), 0) || 0;
  const newLeadsCount = leads?.filter(l => l.status === "new").length || 0;
  const wonDeals = deals?.filter(d => d.stage?.name === "Closed Won") || [];
  const wonValue = wonDeals.reduce((sum, deal) => sum + parseFloat(deal.value), 0);

  const recentLeads = leads?.slice(0, 5) || [];
  const recentDeals = deals?.slice(0, 5) || [];

  const isLoading = leadsLoading || dealsLoading || contactsLoading;

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-semibold" data-testid="text-dashboard-title">Dashboard</h1>
          <p className="text-muted-foreground">Welcome back! Here's your sales overview.</p>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Pipeline Value</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <Skeleton className="h-8 w-24" />
            ) : (
              <div className="text-2xl font-bold" data-testid="text-pipeline-value">
                ${totalPipelineValue.toLocaleString()}
              </div>
            )}
            <p className="text-xs text-muted-foreground">
              Across {deals?.length || 0} active deals
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">New Leads</CardTitle>
            <FileInput className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <Skeleton className="h-8 w-16" />
            ) : (
              <div className="text-2xl font-bold" data-testid="text-new-leads">
                {newLeadsCount}
              </div>
            )}
            <p className="text-xs text-muted-foreground">
              {leads?.length || 0} total leads
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Contacts</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <Skeleton className="h-8 w-16" />
            ) : (
              <div className="text-2xl font-bold" data-testid="text-total-contacts">
                {contacts?.length || 0}
              </div>
            )}
            <p className="text-xs text-muted-foreground">
              In your database
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Won Revenue</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <Skeleton className="h-8 w-24" />
            ) : (
              <div className="text-2xl font-bold text-chart-2" data-testid="text-won-revenue">
                ${wonValue.toLocaleString()}
              </div>
            )}
            <p className="text-xs text-muted-foreground">
              From {wonDeals.length} closed deals
            </p>
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileInput className="h-5 w-5" />
              Recent Leads
            </CardTitle>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="space-y-3">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-12 w-full" />
                ))}
              </div>
            ) : recentLeads.length === 0 ? (
              <div className="flex flex-col items-center justify-center py-8 text-center">
                <FileInput className="h-12 w-12 text-muted-foreground/50 mb-3" />
                <p className="text-muted-foreground">No leads yet</p>
                <p className="text-sm text-muted-foreground/70">Share your lead form to start capturing leads</p>
              </div>
            ) : (
              <div className="space-y-3">
                {recentLeads.map((lead) => (
                  <div 
                    key={lead.id} 
                    className="flex items-center justify-between p-3 rounded-md bg-muted/50"
                    data-testid={`card-lead-${lead.id}`}
                  >
                    <div className="flex flex-col">
                      <span className="font-medium">{lead.firstName} {lead.lastName}</span>
                      <span className="text-sm text-muted-foreground">{lead.email}</span>
                    </div>
                    <Badge 
                      variant={lead.status === "new" ? "default" : "secondary"}
                      className="text-xs"
                    >
                      {lead.status}
                    </Badge>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Handshake className="h-5 w-5" />
              Recent Deals
            </CardTitle>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="space-y-3">
                {[1, 2, 3].map((i) => (
                  <Skeleton key={i} className="h-12 w-full" />
                ))}
              </div>
            ) : recentDeals.length === 0 ? (
              <div className="flex flex-col items-center justify-center py-8 text-center">
                <Handshake className="h-12 w-12 text-muted-foreground/50 mb-3" />
                <p className="text-muted-foreground">No deals yet</p>
                <p className="text-sm text-muted-foreground/70">Create your first deal to get started</p>
              </div>
            ) : (
              <div className="space-y-3">
                {recentDeals.map((deal) => (
                  <div 
                    key={deal.id} 
                    className="flex items-center justify-between p-3 rounded-md bg-muted/50"
                    data-testid={`card-deal-${deal.id}`}
                  >
                    <div className="flex flex-col">
                      <span className="font-medium">{deal.title}</span>
                      <span className="text-sm text-muted-foreground">
                        ${parseFloat(deal.value).toLocaleString()}
                      </span>
                    </div>
                    <Badge 
                      variant="outline"
                      className="text-xs"
                      style={{ borderColor: deal.stage?.color, color: deal.stage?.color }}
                    >
                      {deal.stage?.name}
                    </Badge>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {stages && stages.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="h-5 w-5" />
              Pipeline Overview
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
              {stages.map((stage) => {
                const stageDeals = deals?.filter(d => d.stageId === stage.id) || [];
                const stageValue = stageDeals.reduce((sum, d) => sum + parseFloat(d.value), 0);
                return (
                  <div 
                    key={stage.id} 
                    className="p-4 rounded-md border text-center"
                    data-testid={`card-stage-${stage.id}`}
                  >
                    <div 
                      className="w-3 h-3 rounded-full mx-auto mb-2"
                      style={{ backgroundColor: stage.color }}
                    />
                    <p className="text-sm font-medium truncate">{stage.name}</p>
                    <p className="text-lg font-bold">{stageDeals.length}</p>
                    <p className="text-xs text-muted-foreground">
                      ${stageValue.toLocaleString()}
                    </p>
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
